/**
 * PenguinAttackTCPClient.java
 *
 * Description: Handles sending messages to and receiving messages from the 
 * Penguin Attack server.  After instantiation it is designed to be spun off
 * into its own thread.
 *
 * Copyright (c) 2011, Jason Buck
 * 
 * Distributed under the BSD-new license. For details see the BSD_LICENSE file 
 * that should have been included with this distribution. If the source you 
 * acquired this distribution from incorrectly removed this file, the license 
 * may be viewed at http://www.opensource.org/licenses/bsd-license.php.
 */

import java.util.*;
import java.io.*;
import java.net.*;
import java.util.concurrent.locks.*;

public class PenguinAttackTCPClient implements Runnable
{
    // The specified IP address.
    private InetAddress ipAddress;
    // The port number.
    private static final int PORT = 2323;
    // The socket.
    private Socket socket;
    // The input stream.
    private BufferedReader br;
    // The output stream.
	private PrintWriter out;
	// Are we connected?
	private boolean connected = false;
	// Dead simple locking mechanism
	private Lock lock = new ReentrantLock(true);
	// Most recently received board state
	private String[] boardState;
	// Flag indicating that we have received a new board state
	private boolean receivedNewBoard = false;
	// Most recently received cursor position
	private String cursorPosition;
	// Flag indicating that we have received a new cursor position
	private boolean receivedNewCursor = false;
	// Flag indicating that we have received a new timer
	private boolean receivedNewTimer;
	// Most recently received timer
	private String timer;
	private boolean gameHasStarted;
	private boolean gameIsOver;
	private boolean aiWon;
	private boolean gamePaused;
		
    /**
     * Constructor initializes a new PenguinAttackTCPClient object. Sets up  
	 * networking components and connects to the server.
	 *
     * @param ipString The specified address of the Penguin Attack Server.
     */
    public PenguinAttackTCPClient(String ipString)
    {
        this.ipAddress = null;
        this.socket = null;
        this.br = null;
		
        // Attempt to initialize the instance variables.
        try
        {
            this.ipAddress = InetAddress.getByName(ipString);
            this.socket = new Socket(ipAddress, PORT);
            this.br = new BufferedReader(new InputStreamReader(socket.getInputStream()));
			this.out = new PrintWriter(socket.getOutputStream(), true);
        } 
		catch (Exception e)
        {
            return;
        }
		
		this.connected = true;
	}
	
	/*
	 * Runnable interface implementation
	 * Don't call this:  Create a new thread with an object of this class as a
	 * parameter, then call Thread.start().
	 */
	public void run()
	{
		this.readData();
		this.disconnectClient();
	}
	
	/**
	 * Accessor used to check whether or not the constructor successfully made a
	 * connection.
	 *
	 * @return boolean	True if the constructor successfully connected.
	 */
	public boolean isConnected()
	{
		return this.connected;
	}
	
	/*
	 * Method to shut down the networking components.
	 */
	private void disconnectClient()
	{
        try
        {
			this.out.close();
            this.br.close();
			this.connected = false;
        }
		catch (Exception e)
        {
            System.out.println("Exception caught while trying to close the connection:  " + e.getMessage());
            System.exit(1);
        }
	}
	
    /*
     * Method to get data from the Penguin Attack Server and store a completed
	 * message in the appropriate instance variable.
     */
    private void readData()
    {
        String message = "";
		// Should we keep reading from the network?
		boolean keepReading = true;
		boolean receivingBoardState = false;
		ArrayList<String> messageBuffer = new ArrayList<String>();

        while (keepReading)
        {
			// Read the data from the input stream.
			try
			{
				message = this.br.readLine();
				if(message.equals(""))	// not sure this if statement is really
					continue;			// necessary, but once was convinced.
			} 
			catch (Exception e)
			{
				System.out.println("Penguin Attack server closed the connection (abrupt).");
				this.connected = false;
				return;
			}
				
			// Has the game started?
			if (message.contains("start"))
			{
				this.lock.lock();
				try 
				{
					this.gameHasStarted = true;
					this.gameIsOver = false;
					this.aiWon = false;
				}
				finally 
				{
					this.lock.unlock();
				}
				continue;
			}
			
			// Has the game been paused?
			if (message.contains("paused"))
			{
				this.lock.lock();
				try 
				{
					this.gameHasStarted = false;
				}
				finally 
				{
					this.lock.unlock();
				}
				continue;
			}
			
			// Did we win?
			if(message.contains("win"))
			{
				this.lock.lock();
				try 
				{
					this.gameIsOver = true;
					this.aiWon = true;
					this.gameHasStarted = false;
				}
				finally 
				{
					this.lock.unlock();
				}
				continue;
			}
			
			// Did we lose?
			if(message.contains("lose"))
			{
				this.lock.lock();
				try 
				{
					this.gameIsOver = true;
					this.aiWon = false;
					this.gameHasStarted = false;
				}
				finally 
				{
					this.lock.unlock();
				}
				continue;
			}
			
			// Are we done here?
			if (message.contains("closing connection") || message.equals("pa_network: closing connection"))
			{
				System.out.println("Penguin Attack server closed the connection (graceful).");
				keepReading = false;
				continue;
			}
			
			// Is this the beginning of a board state?
			if (message.contains("b{"))
			{	
				receivingBoardState = true;
				//System.out.println("Receiving board state");
				continue;
			}
			
			// Is this the end of a message?
			if (message.contains("}"))
			{
				if (receivingBoardState)
				{
					receivingBoardState = false;
					this.lock.lock();
					try 
					{
						this.boardState = new String[messageBuffer.size()];
						this.boardState = messageBuffer.toArray(boardState);
						messageBuffer.clear();
						this.receivedNewBoard = true;
					}
					finally 
					{
						this.lock.unlock();
					}
				}
				continue;
			}
			
			// Is this a cursor position message?
			if (message.charAt(0) == '<')
			{
				this.lock.lock();
				try 
				{
					this.cursorPosition = message;
					this.receivedNewCursor = true;
				}
				finally 
				{
					this.lock.unlock();
				}
				continue;
			}
			
			// Is this a timer message?
			if(message.charAt(0) == '[')
			{
				this.lock.lock();
				try 
				{
					this.timer = message;
					this.receivedNewTimer = true;
				}
				finally 
				{
					this.lock.unlock();
				}
				continue;
			}
			
			messageBuffer.add(message);
        }
    }
    
	/**
	 * Accessor:  Tells if the game has started.
	 *
	 * @return boolean True if the game has started, false otherwise.
	 */
    public boolean gameStarted()
    {
    	this.lock.lock();
		try 
		{
			return gameHasStarted;
		}
		finally 
		{
			this.lock.unlock();
		}
    }
    
	/**
	 * Accessor:  Tells if the game is over.
	 *
	 * @return boolean True if the game is over, false otherwise.
	 */
    public boolean gameOver()
    {
    	this.lock.lock();
		try 
		{
			return this.gameIsOver;
		}
		finally 
		{
			this.lock.unlock();
		}
    }
    
	/**
	 * Mutator: Resets the game for another round.
	 */
    public void resetGameOver()
    {
    	this.gameIsOver = false;
    }
    
	/**
	 * Accessor: Find out if the AI won or not.
	 *
	 * @return boolean True if the AI won the game, false otherwise.
	 */
    public boolean aiIsWinner()
    {
    	this.lock.lock();
		try 
		{
			return this.aiWon;
		}
		finally 
		{
			this.lock.unlock();
		}
    }
	
	/**
	 * Accessor: check and see if we have completed receiving a new board state.
	 *
	 * @return boolean true if we have finished receiving a new board state.
	 */
	public boolean receivedNewBoardState()
	{	
		boolean temp;
		this.lock.lock();
		try 
		{
			temp = this.receivedNewBoard;
		}
		finally 
		{
			this.lock.unlock();
		}
		
		return temp;
	}
	
	/**
	 * Accessor: check and see if we have received a new cursor position.
	 *
	 * @return boolean true if we have received a new cursor position.
	 */
	public boolean receivedNewCursorPosition()
	{
		boolean temp;
		this.lock.lock();
		try 
		{
			temp = this.receivedNewCursor;
		}
		finally 
		{
			this.lock.unlock();
		}
		
		return temp;
	}
	
	/**
	 * Accessor: check and see if we have received a new timer state.
	 *
	 * @return boolean true if we have received a new timer state.
	 */
	public boolean receivedNewTimerState()
	{
		boolean temp;
		this.lock.lock();
		try 
		{
			temp = this.receivedNewTimer;
		}
		finally 
		{
			this.lock.unlock();
		}
		
		return temp;
	}
	
	/**
	 * Sends a sequence of moves given as a string.  Characters are parsed and
	 * sent individually, with a brief delay between moves (we end up trying to 
	 * swap blocks that are still swapping if we send them all at once).
	 *
	 * @param sequence	A sequence of PA moves.
	 * u = up
	 * d = down
	 * l = left
	 * r = right
	 * s = swap
	 * a = raise stack
	 */
	public void sendMoveSequence(String sequence)
	{
		if(sequence.indexOf('s') == -1)
		{
			this.out.println(sequence);
			return;
		}
		else
			this.out.println(sequence.substring(0, sequence.indexOf('s')));
			
		for (int i = sequence.indexOf('s'); i < sequence.length(); i++) 
		{
			if(this.gameIsOver || !this.connected)
				return;
			
			this.out.println(String.valueOf(sequence.charAt(i)));
			
			try 
			{
				if(this.gameIsOver || !this.connected || this.gamePaused)
					return;
				Thread.sleep(100);
			}
			catch (Exception e) {}	
		}
	}
	
	/**
	 * Sends the PA server a command.
	 *
	 * @param command	A command.
	 * 'b' = request board state
	 * 'quit' = signal intention to disconnect from server
	 */
	public void sendCommand(String command)
	{
		if(this.gameIsOver || !this.connected)
			return;
		this.out.println(command);
	}
	
	/**
	 * Acquires the most recent board state and sets receivedNewBoard to false.
	 *
	 * @return String[]	The most recently received board state strings.
	 */
	public String[] getBoardState()
	{
		String[] temp = null;
		
		this.lock.lock();
		try 
		{
			temp = cloneArray(this.boardState);
			this.receivedNewBoard = false;
		}
		finally 
		{
			this.lock.unlock();
		}
		
		return temp;
	}
	
	/**
	 * Acquires the most recent cursor position and sets receivedNewCursor to
	 * false.
	 *
	 * @return String[]	The most recently received board state strings.
	 */
	public String getCursorPosition()
	{
		String temp = "";
		this.lock.lock();
		try 
		{
			temp = this.cursorPosition;
			this.receivedNewCursor = false;
		}
		finally 
		{
			this.lock.unlock();
		}
		
		return temp;
	}
	
	/**
	 * Acquires the most recent timer and sets receivedNewTimer to
	 * false.
	 *
	 * @return String[]	The most recently received timer state strings.
	 */
	public String getTimer()
	{
		String temp = "";
		this.lock.lock();
		try 
		{
			temp = this.timer;
			this.receivedNewTimer = false;
		}
		finally 
		{
			this.lock.unlock();
		}
		
		return temp;
	}
	
	/**
	 * Clones a one-dimensional String array.  Copies the values stored in the
	 * parameter into the new returned array.
	 * 
	 * @param array - Could be any one-dimensional String array.
	 * @return String[] - A copy of the parameter array.
	 */
	public static String[] cloneArray(String[] array)
	{
		String[] newArray = new String[array.length];
		for (int i = 0; i < array.length; i++)
			newArray[i] = array[i];
		
		return newArray;
	}
	
}